#include "xparameters.h"
#include "xgpio.h"

#include <sleep.h>
#include <errno.h>

#define CMD_FLASH_RESET  0xff
#define CMD_GET_FEATURES 0xee
#define CMD_SET_FEATURES 0xef
#define CMD_READ_ID      0x90

#define CMD_PROGRAM_MODE 0x80
#define CMD_PROGRAM_PAGE 0x10

#define CMD_READ_MODE 0x00
#define CMD_READ_PAGE 0x30

#define CMD_ERASE_MODE  0x60
#define CMD_ERASE_BLOCK 0xd0

#define CMD_READ_STATUS_ENHANCED 0x78

#define SR_FAIL   (1 << 0)
#define SR_FAILC  (1 << 1)
#define SR_ARDY   (1 << 5)
#define SR_RDY    (1 << 6)
#define SR_WP     (1 << 7)
#define SR_ALLRDY (SR_ARDY | SR_RDY | SR_WP)

#define CLE (1 << 0)
#define ALE (1 << 1)
#define WE  (1 << 2)
#define RE_T  (1 << 3)
#define RE_C  (1 << 4)
#define DQS_T (1 << 5)
#define DQS_C (1 << 6)

#define pack_address pack_address_micron

static void write_command(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                          u8 command)
{
    XGpio_SetDataDirection(data_gpio, channel, 0x00);
    /* A command is written from DQ[7:0] to the command register on the rising
       edge of WE# when CE# is LOW, ALE is LOW, CLE is HIGH, and RE# is HIGH.
     */
    XGpio_DiscreteWrite(cmd_gpio, channel, DQS_T | RE_T | CLE);
    XGpio_DiscreteWrite(data_gpio, channel, command);
    XGpio_DiscreteWrite(cmd_gpio, channel, DQS_T | RE_T | WE | CLE);
}

static void write_address(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                          u8 addr)
{
    XGpio_SetDataDirection(data_gpio, channel, 0x00);
    /* A NV-DDR2 address is written from DQ[7:0] to the address register on the
       rising edge of WE# when CE# is LOW, ALE is HIGH, CLE is LOW, and RE# is
       HIGH.
     */
    XGpio_DiscreteWrite(cmd_gpio, channel, DQS_T | RE_T | ALE);
    XGpio_DiscreteWrite(data_gpio, channel, addr);
    XGpio_DiscreteWrite(cmd_gpio, channel, DQS_T | RE_T | WE | ALE);
    /* XGpio_DiscreteWrite(cmd_gpio, channel, 0b01111); */
}

static void data_out_transfer(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                              u8 data)
{
    XGpio_SetDataDirection(data_gpio, channel, 0x00);
    XGpio_DiscreteWrite(cmd_gpio, channel, DQS_T | RE_T);
    XGpio_DiscreteWrite(data_gpio, channel, data);
    XGpio_DiscreteWrite(cmd_gpio, channel, DQS_T | RE_T | WE);
}

static void data_in_transfer_sync(XGpio* cmd_gpio, XGpio* data_gpio,
                                  int channel, u8* data, size_t size)
{
    u8 cmd = WE;
    u32 offset = ((channel - 1) * XGPIO_CHAN_OFFSET) + XGPIO_DATA_OFFSET;
    int i;

    XGpio_SetDataDirection(data_gpio, channel, 0xff);
    XGpio_SetDataDirection(cmd_gpio, channel, DQS_T);
    XGpio_DiscreteWrite(cmd_gpio, channel, cmd);
    for (i = 0; i < size; i++) {
        cmd = cmd ^ RE_T;
        XGpio_WriteReg(cmd_gpio->BaseAddress, offset, cmd);
        data[i] = XGpio_ReadReg(data_gpio->BaseAddress, offset);
    }
    XGpio_DiscreteWrite(cmd_gpio, channel, WE);
    XGpio_SetDataDirection(cmd_gpio, channel, 0);
}

static void data_out_transfer_sync(XGpio* cmd_gpio, XGpio* data_gpio,
                                   int channel, const u8* data, size_t size)
{
    u8 cmd = RE_T | WE;
    int i;

    XGpio_SetDataDirection(data_gpio, channel, 0x00);
    XGpio_DiscreteWrite(cmd_gpio, channel, cmd);
    for (i = 0; i < size; i++) {
        XGpio_DiscreteWrite(data_gpio, channel, data[i]);
        cmd = cmd ^ DQS_T;
        XGpio_DiscreteWrite(cmd_gpio, channel, cmd);
    }
    XGpio_DiscreteWrite(cmd_gpio, channel, RE_T | WE);
}

static void pack_address_micron(unsigned int die, unsigned int plane,
                                unsigned int block, unsigned int page,
                                unsigned col, u8* buf)
{
    /* MLC p33 */
    buf[0] = col & 0xff;
    buf[1] = (col >> 8) & 0x7f;
    buf[2] = page & 0xff;
    buf[3] = ((page >> 8) & 1) | ((plane & 1) << 1) | ((block & 0x3f) << 2);
    buf[4] = ((block >> 6) & 0x1f) | ((die & 1) << 5);
}

static void address_cycle(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                          unsigned int die, unsigned int plane,
                          unsigned int block, unsigned int page, unsigned col)
{
    u8 buf[5];
    int i;

    pack_address(die, plane, block, page, col, buf);
    for (i = 0; i < 5; i++)
        write_address(cmd_gpio, data_gpio, channel, buf[i]);
}

void fcmd_reset(XGpio* cmd_gpio, XGpio* data_gpio, int channel)
{
    write_command(cmd_gpio, data_gpio, channel, CMD_FLASH_RESET);
}

void fcmd_activate_nvddr2(XGpio* cmd_gpio, XGpio* data_gpio, int channel)
{
    write_command(cmd_gpio, data_gpio, channel, CMD_SET_FEATURES);
    write_address(cmd_gpio, data_gpio, channel, 0x01);
    data_out_transfer(cmd_gpio, data_gpio, channel, 0x27);
    data_out_transfer(cmd_gpio, data_gpio, channel, 0x00);
    data_out_transfer(cmd_gpio, data_gpio, channel, 0x00);
    data_out_transfer(cmd_gpio, data_gpio, channel, 0x00);
}

void fcmd_read_id(XGpio* cmd_gpio, XGpio* data_gpio, int channel, u8 buf[10])
{
    write_command(cmd_gpio, data_gpio, channel, CMD_READ_ID);
    write_address(cmd_gpio, data_gpio, channel, 0x00);

    data_in_transfer_sync(cmd_gpio, data_gpio, channel, buf, 10);
}

static u8 read_status(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                      unsigned int die, unsigned int plane)
{
    u8 status;
    u8 buf[5];
    int i;

    pack_address(die, plane, 0, 0, 0, buf);
    write_command(cmd_gpio, data_gpio, channel, CMD_READ_STATUS_ENHANCED);
    for (i = 2; i < 5; i++)
        write_address(cmd_gpio, data_gpio, channel, buf[i]);
    XGpio_SetDataDirection(data_gpio, channel, 0xff);
    XGpio_DiscreteWrite(cmd_gpio, channel, RE_T | WE);
    XGpio_DiscreteWrite(cmd_gpio, channel, WE);
    status = XGpio_DiscreteRead(data_gpio, channel);

    return status;
}

static int wait_status(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                       unsigned int die, unsigned int plane, int expected)
{
    u8 status;
    int count = 0;

    for (;;) {
        status = read_status(cmd_gpio, data_gpio, channel, die, plane);

        if (status == expected) return 0;

        if (status & 0x3) return EIO;

        usleep(5);
        count += 10;
        if (count > 12000) return ETIMEDOUT;
    }

    return 0;
}

u8 fcmd_is_ready(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                 unsigned int die, int* error)
{
    u8 status = read_status(cmd_gpio, data_gpio, channel, die, 0);

    if (error) {
        *error = !!(status & (SR_FAIL | SR_FAILC));
    }

    return status == SR_ALLRDY;
}

int fcmd_wait_ready(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                    unsigned int die, unsigned int plane)
{
    return wait_status(cmd_gpio, data_gpio, channel, die, plane, SR_ALLRDY);
}

void fcmd_read_page(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                    unsigned int die, unsigned int plane, unsigned int block,
                    unsigned int page, unsigned int col)
{
    write_command(cmd_gpio, data_gpio, channel, CMD_READ_MODE);
    address_cycle(cmd_gpio, data_gpio, channel, die, plane, block, page, col);
    write_command(cmd_gpio, data_gpio, channel, CMD_READ_PAGE);
}

void fcmd_read_page_data_sync(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                              unsigned int die, unsigned int plane,
                              unsigned int block, unsigned int page,
                              unsigned int col, u8* data, size_t size)
{
    write_command(cmd_gpio, data_gpio, channel, CMD_READ_MODE);
    data_in_transfer_sync(cmd_gpio, data_gpio, channel, data, size);
}

void fcmd_program_page_sync(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                            unsigned int die, unsigned int plane,
                            unsigned int block, unsigned int page,
                            unsigned int col, const u8* data, size_t size)
{
    write_command(cmd_gpio, data_gpio, channel, CMD_PROGRAM_MODE);
    address_cycle(cmd_gpio, data_gpio, channel, die, plane, block, page, col);
    data_out_transfer_sync(cmd_gpio, data_gpio, channel, data, size);
    write_command(cmd_gpio, data_gpio, channel, CMD_PROGRAM_PAGE);
}

void fcmd_erase_block(XGpio* cmd_gpio, XGpio* data_gpio, int channel,
                      unsigned int die, unsigned int plane, unsigned int block)
{
    u8 buf[5];
    int i;

    pack_address(die, plane, block, 0, 0, buf);
    write_command(cmd_gpio, data_gpio, channel, CMD_ERASE_MODE);
    for (i = 2; i < 5; i++)
        write_address(cmd_gpio, data_gpio, channel, buf[i]);
    write_command(cmd_gpio, data_gpio, channel, CMD_ERASE_BLOCK);
}
