#include <stdio.h>
#include "xil_printf.h"
#include "xil_assert.h"
#include "xparameters.h"
#include "xgpio.h"
#include "xgpiops.h"
#include <sleep.h>
#include <stddef.h>

#include "fcmd.h"

#include "bch_engine.h"

#define NR_CHANNELS 4
#define NR_CHIPS 2

#define CMD_GPIO(chan) (&cmd_gpios[(chan)>>1])
#define DATA_GPIO(chan) (&data_gpios[(chan)>>1])
#define SUB_INDEX(chan) (((chan)&1)+1)

#define CHIP_INDEX(chan, chip) ((chan) * NR_CHIPS + (chip))

#define CE(chan, chip) ce_pins[CHIP_INDEX(chan, chip)]

#define BCH_BLOCK_SIZE (512)
#define BCH_CODE_SIZE  (20)

static XGpioPs ps_gpio_inst;
static XGpio cmd_gpios[NR_CHANNELS];
static XGpio data_gpios[NR_CHANNELS];

static struct bch_engine bch_engine;

static int cmd_gpio_ids[] = {
    XPAR_A_COMMAND_DEVICE_ID,
    XPAR_B_COMMAND_DEVICE_ID,
    XPAR_C_COMMAND_DEVICE_ID,
    XPAR_D_COMMAND_DEVICE_ID,
};

static int data_gpio_ids[] = {
    XPAR_A_DATA_DEVICE_ID,
    XPAR_B_DATA_DEVICE_ID,
    XPAR_C_DATA_DEVICE_ID,
    XPAR_D_DATA_DEVICE_ID,
};

//static int ce_pins[] = {
//        36, 20, 30, 27, \
//            22, 28, 37, 43, \
//            63, 58, 31, 21, \
//            57, 65, 38, 54, \
//            67, 64, 72, 61, \
//            60, 66, 76, 73, \
//            74, 71, 56, 59, \
//            70, 69, 55, 75,
//};

static int ce_pins[] = {
		30, 27, \
          37, 43, \
          31, 21, \
          38, 54, \
          67, 64, \
          60, 66, \
          56, 59, \
          55, 75,
};

static int wp_pins[] = {
        29, 26, 52, 53, 14, 24, 62, 68
};

static unsigned long lsb_bitmap[] = {
    0xCCCCCCCCCCCCCDBFUL, 0xCCCCCCCCCCCCCCCCUL, 0xCCCCCCCCCCCCCCCCUL, 0xCCCCCCCCCCCCCCCCUL, 0xCCCCCCCCCCCCCCCCUL, 0xCCCCCCCCCCCCCCCCUL, 0xCCCCCCCCCCCCCCCCUL, 0x6CCCCCCCCCCCCCCCUL
};

static inline int is_lsb(int page)
{
    return !!(lsb_bitmap[page >> 6] & (1UL << (page & 0x3f)));
}

static inline void write_ce(int pin, int enable)
{
    XGpioPs_WritePin(&ps_gpio_inst, pin, !enable);
}

static void init_gpio(void)
{
    XGpioPs_Config* gpio_config_ptr;
    int i, j, status;

    gpio_config_ptr = XGpioPs_LookupConfig(XPAR_PSU_GPIO_0_DEVICE_ID);
    if (gpio_config_ptr == NULL) {
        xil_printf("PS GPIO not found\n\r");
        return;
    }

    status = XGpioPs_CfgInitialize(&ps_gpio_inst, gpio_config_ptr,
                                   gpio_config_ptr->BaseAddr);
    if (status != XST_SUCCESS) {
        xil_printf("PS GPIO init failed\n\r");
        return;
    }

    for (i = 0; i < sizeof(ce_pins) / sizeof(ce_pins[0]); i++) {
        XGpioPs_SetDirectionPin(&ps_gpio_inst, ce_pins[i], 1);
        XGpioPs_SetOutputEnablePin(&ps_gpio_inst, ce_pins[i], 1);
        XGpioPs_WritePin(&ps_gpio_inst, ce_pins[i], 1);
    }

    for (i = 0; i < sizeof(wp_pins) / sizeof(wp_pins[0]); i++) {
        XGpioPs_SetDirectionPin(&ps_gpio_inst, wp_pins[i], 1);
        XGpioPs_SetOutputEnablePin(&ps_gpio_inst, wp_pins[i], 1);
        XGpioPs_WritePin(&ps_gpio_inst, wp_pins[i], 1);
    }

    for (i = 0; i < NR_CHANNELS; i++) {
        XGpio* cmd_gpio = &cmd_gpios[i];
        XGpio* data_gpio = &data_gpios[i];

        status = XGpio_Initialize(cmd_gpio, cmd_gpio_ids[i]);
        if (status != XST_SUCCESS) {
            xil_printf("Command GPIO for channel %d init failed\n\r", i);
            return;
        }

        status = XGpio_Initialize(data_gpio, data_gpio_ids[i]);
        if (status != XST_SUCCESS) {
            xil_printf("Data GPIO for channel %d init failed\n\r", i);
            return;
        }

        for (j = 0; j < 2; j++) {
            XGpio_SetDataDirection(cmd_gpio, j + 1, 0);
        }
    }
}


static void reset_flash(void)
{
    int i, j;

    for (i = 0; i < 8; i++) {
        for (j = 0; j < 4; j++) {
            write_ce(CE(i, j), TRUE);
            fcmd_reset(CMD_GPIO(i), DATA_GPIO(i), SUB_INDEX(i));
            write_ce(CE(i, j), FALSE);
        }

    }
    usleep(2000);

    for (i = 0; i < 8; i++) {
        for (j = 0; j < 4; j++) {
            write_ce(CE(i, j), TRUE);
            fcmd_activate_nvddr2(CMD_GPIO(i), DATA_GPIO(i), SUB_INDEX(i));
            write_ce(CE(i, j), FALSE);
        }
    }
    usleep(1);
}

static void read_id_simple(unsigned int channel, unsigned int chip, u8 buf[10]) {
	write_ce(CE(channel, chip), TRUE);
	fcmd_read_id(CMD_GPIO(channel), DATA_GPIO(channel), SUB_INDEX(channel), buf);
	write_ce(CE(channel, chip), FALSE);
}

static void read_page_simple(unsigned int channel, unsigned int chip,
                            unsigned int die, unsigned int plane,
                            unsigned int block, unsigned int page,
                            unsigned int col, u8* buffer, size_t count)
{
    write_ce(CE(channel, chip), TRUE);
    fcmd_read_page(CMD_GPIO(channel), DATA_GPIO(channel), SUB_INDEX(channel), die, plane, block, page, col);
    fcmd_wait_ready(CMD_GPIO(channel), DATA_GPIO(channel), SUB_INDEX(channel), die, plane);
    fcmd_read_page_data_sync(CMD_GPIO(channel), DATA_GPIO(channel), SUB_INDEX(channel), die, plane, block, page, col, buffer, count);
    write_ce(CE(channel, chip), FALSE);
}

static void program_page_simple(unsigned int channel, unsigned int chip,
                            unsigned int die, unsigned int plane,
                            unsigned int block, unsigned int page,
                            unsigned int col, const u8* buffer, size_t count)
{
    write_ce(CE(channel, chip), TRUE);
    fcmd_program_page_sync(CMD_GPIO(channel), DATA_GPIO(channel), SUB_INDEX(channel), die, plane, block, page, col, buffer, count);
    fcmd_wait_ready(CMD_GPIO(channel), DATA_GPIO(channel), SUB_INDEX(channel), die, plane);
    write_ce(CE(channel, chip), FALSE);
}

static void erase_block_simple(unsigned int channel, unsigned int chip,
                            unsigned int die, unsigned int plane,
                            unsigned int block)
{
    write_ce(CE(channel, chip), TRUE);
    fcmd_erase_block(CMD_GPIO(channel), DATA_GPIO(channel), SUB_INDEX(channel), die, plane, block);
    fcmd_wait_ready(CMD_GPIO(channel), DATA_GPIO(channel), SUB_INDEX(channel), die, plane);
    write_ce(CE(channel, chip), FALSE);
}

static int check_page(unsigned char* buf, unsigned char* gt) {
    int err = 0;
    int i, j;

    for (i = 0; i < 32; i++) {
        int start = i * 532;
        for (j = 0; j < 512; j++) {
            if (buf[start+j] != gt[i*512+j]) {
                err++;
            }
        }
    }

    return err;
}

static void encode_page(const u8* in_buf, u8* out_buf)
{
    int i, j;

    for (i = 0; i < 32; i++) {
        memcpy(&out_buf[i*532], &in_buf[i*512], 512);
        bch_engine_calculate(&bch_engine, &out_buf[i*532], &out_buf[i*532+512]);
    }
}

int main()
{
    static unsigned char out_page_buf[0x6000];
    static unsigned char in_page_buf[0x6000];
    static unsigned char data_buf[0x4000];
    int i, j, k;
    int err;
    int r;
    u8 id_buf[10];

//    init_gpio();
//
//    reset_flash();

    r = bch_engine_init(&bch_engine, BCH_BLOCK_SIZE, BCH_CODE_SIZE);
    if (r != 0) {
        xil_printf("Failed to initialize BCH engine (%d)\n", r);
        return XST_FAILURE;
    }

    for (i = 0; i < 0x4000; i++)
        data_buf[i] = i & 0xff;
    encode_page(data_buf, out_page_buf);

    for (i = 512; i < 532; i++)
    	xil_printf("%02x ", out_page_buf[i]);
    xil_printf("\n");

//    read_id_simple(0, 0, id_buf);
//    for (i = 0; i < 10; i++) {
//    	xil_printf("%02x ", id_buf[i]);
//    }
//    xil_printf("\n");

//    erase_block_simple(4, 0, 1, 0, 1046);
//    program_page_simple(4, 0, 1, 0, 1046, 330, 0, out_page_buf, 16384 + 32 * 20);
//    usleep(100);
//    read_page_simple(4, 0, 1, 0, 1046, 330, 0, in_page_buf, 16384 + 32 * 20);
//
//    for (i = 0; i < 32; i ++) {
//        int j, k;
//
//        for (j = 0; j < 512; j+=32) {
//            for (k = 0; k < 32; k++)
//                xil_printf("%x ", in_page_buf[i*532 + j+k]);
//            xil_printf("\n");
//        }
//
//        xil_printf(">> ");
//        for (k = 0; k < 20; k++)
//           xil_printf("%x ", in_page_buf[i*532+512+k]);
//
//        xil_printf("\n");
//    }
//
//    for (i = 0; i < 32; i++) {
//        int start = i * 532;
//
//        bch_engine_calculate(&bch_engine, &in_page_buf[start], data_buf);
//        int a = bch_engine_correct(&bch_engine, &in_page_buf[start], &in_page_buf[start+512], data_buf);
//    }

    return 0;
}
